// Copyright 2026 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
// ** This file is automatically generated by gapic-generator-typescript. **
// ** https://github.com/googleapis/gapic-generator-typescript **
// ** All changes to this file may be overwritten. **

import * as protos from '../protos/protos';
import * as assert from 'assert';
import * as sinon from 'sinon';
import {SinonStub} from 'sinon';
import {describe, it} from 'mocha';
import * as spannerModule from '../src';

import {PassThrough} from 'stream';

import {protobuf} from 'google-gax';

// Dynamically loaded proto JSON is needed to get the type information
// to fill in default values for request objects
const root = protobuf.Root.fromJSON(
  require('../protos/protos.json'),
).resolveAll();

// eslint-disable-next-line @typescript-eslint/no-unused-vars
function getTypeDefaultValue(typeName: string, fields: string[]) {
  let type = root.lookupType(typeName) as protobuf.Type;
  for (const field of fields.slice(0, -1)) {
    type = type.fields[field]?.resolvedType as protobuf.Type;
  }
  return type.fields[fields[fields.length - 1]]?.defaultValue;
}

function generateSampleMessage<T extends object>(instance: T) {
  const filledObject = (
    instance.constructor as typeof protobuf.Message
  ).toObject(instance as protobuf.Message<T>, {defaults: true});
  return (instance.constructor as typeof protobuf.Message).fromObject(
    filledObject,
  ) as T;
}

function stubSimpleCall<ResponseType>(response?: ResponseType, error?: Error) {
  return error
    ? sinon.stub().rejects(error)
    : sinon.stub().resolves([response]);
}

function stubSimpleCallWithCallback<ResponseType>(
  response?: ResponseType,
  error?: Error,
) {
  return error
    ? sinon.stub().callsArgWith(2, error)
    : sinon.stub().callsArgWith(2, null, response);
}

function stubServerStreamingCall<ResponseType>(
  response?: ResponseType,
  error?: Error,
) {
  const transformStub = error
    ? sinon.stub().callsArgWith(2, error)
    : sinon.stub().callsArgWith(2, null, response);
  const mockStream = new PassThrough({
    objectMode: true,
    transform: transformStub,
  });
  // write something to the stream to trigger transformStub and send the response back to the client
  setImmediate(() => {
    mockStream.write({});
  });
  setImmediate(() => {
    mockStream.end();
  });
  return sinon.stub().returns(mockStream);
}

function stubPageStreamingCall<ResponseType>(
  responses?: ResponseType[],
  error?: Error,
) {
  const pagingStub = sinon.stub();
  if (responses) {
    for (let i = 0; i < responses.length; ++i) {
      pagingStub.onCall(i).callsArgWith(2, null, responses[i]);
    }
  }
  const transformStub = error
    ? sinon.stub().callsArgWith(2, error)
    : pagingStub;
  const mockStream = new PassThrough({
    objectMode: true,
    transform: transformStub,
  });
  // trigger as many responses as needed
  if (responses) {
    for (let i = 0; i < responses.length; ++i) {
      setImmediate(() => {
        mockStream.write({});
      });
    }
    setImmediate(() => {
      mockStream.end();
    });
  } else {
    setImmediate(() => {
      mockStream.write({});
    });
    setImmediate(() => {
      mockStream.end();
    });
  }
  return sinon.stub().returns(mockStream);
}

function stubAsyncIterationCall<ResponseType>(
  responses?: ResponseType[],
  error?: Error,
) {
  let counter = 0;
  const asyncIterable = {
    [Symbol.asyncIterator]() {
      return {
        async next() {
          if (error) {
            return Promise.reject(error);
          }
          if (counter >= responses!.length) {
            return Promise.resolve({done: true, value: undefined});
          }
          return Promise.resolve({done: false, value: responses![counter++]});
        },
      };
    },
  };
  return sinon.stub().returns(asyncIterable);
}

describe('v1.SpannerClient', () => {
  describe('Common methods', () => {
    it('has apiEndpoint', () => {
      const client = new spannerModule.v1.SpannerClient();
      const apiEndpoint = client.apiEndpoint;
      assert.strictEqual(apiEndpoint, 'spanner.googleapis.com');
    });

    it('has universeDomain', () => {
      const client = new spannerModule.v1.SpannerClient();
      const universeDomain = client.universeDomain;
      assert.strictEqual(universeDomain, 'googleapis.com');
    });

    if (
      typeof process === 'object' &&
      typeof process.emitWarning === 'function'
    ) {
      it('throws DeprecationWarning if static servicePath is used', () => {
        const stub = sinon.stub(process, 'emitWarning');
        const servicePath = spannerModule.v1.SpannerClient.servicePath;
        assert.strictEqual(servicePath, 'spanner.googleapis.com');
        assert(stub.called);
        stub.restore();
      });

      it('throws DeprecationWarning if static apiEndpoint is used', () => {
        const stub = sinon.stub(process, 'emitWarning');
        const apiEndpoint = spannerModule.v1.SpannerClient.apiEndpoint;
        assert.strictEqual(apiEndpoint, 'spanner.googleapis.com');
        assert(stub.called);
        stub.restore();
      });
    }
    it('sets apiEndpoint according to universe domain camelCase', () => {
      const client = new spannerModule.v1.SpannerClient({
        universeDomain: 'example.com',
      });
      const servicePath = client.apiEndpoint;
      assert.strictEqual(servicePath, 'spanner.example.com');
    });

    it('sets apiEndpoint according to universe domain snakeCase', () => {
      const client = new spannerModule.v1.SpannerClient({
        universe_domain: 'example.com',
      });
      const servicePath = client.apiEndpoint;
      assert.strictEqual(servicePath, 'spanner.example.com');
    });

    if (typeof process === 'object' && 'env' in process) {
      describe('GOOGLE_CLOUD_UNIVERSE_DOMAIN environment variable', () => {
        it('sets apiEndpoint from environment variable', () => {
          const saved = process.env['GOOGLE_CLOUD_UNIVERSE_DOMAIN'];
          process.env['GOOGLE_CLOUD_UNIVERSE_DOMAIN'] = 'example.com';
          const client = new spannerModule.v1.SpannerClient();
          const servicePath = client.apiEndpoint;
          assert.strictEqual(servicePath, 'spanner.example.com');
          if (saved) {
            process.env['GOOGLE_CLOUD_UNIVERSE_DOMAIN'] = saved;
          } else {
            delete process.env['GOOGLE_CLOUD_UNIVERSE_DOMAIN'];
          }
        });

        it('value configured in code has priority over environment variable', () => {
          const saved = process.env['GOOGLE_CLOUD_UNIVERSE_DOMAIN'];
          process.env['GOOGLE_CLOUD_UNIVERSE_DOMAIN'] = 'example.com';
          const client = new spannerModule.v1.SpannerClient({
            universeDomain: 'configured.example.com',
          });
          const servicePath = client.apiEndpoint;
          assert.strictEqual(servicePath, 'spanner.configured.example.com');
          if (saved) {
            process.env['GOOGLE_CLOUD_UNIVERSE_DOMAIN'] = saved;
          } else {
            delete process.env['GOOGLE_CLOUD_UNIVERSE_DOMAIN'];
          }
        });
      });
    }
    it('does not allow setting both universeDomain and universe_domain', () => {
      assert.throws(() => {
        new spannerModule.v1.SpannerClient({
          universe_domain: 'example.com',
          universeDomain: 'example.net',
        });
      });
    });

    it('has port', () => {
      const port = spannerModule.v1.SpannerClient.port;
      assert(port);
      assert(typeof port === 'number');
    });

    it('should create a client with no option', () => {
      const client = new spannerModule.v1.SpannerClient();
      assert(client);
    });

    it('should create a client with gRPC fallback', () => {
      const client = new spannerModule.v1.SpannerClient({
        fallback: true,
      });
      assert(client);
    });

    it('has initialize method and supports deferred initialization', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      assert.strictEqual(client.spannerStub, undefined);
      await client.initialize();
      assert(client.spannerStub);
    });

    it('has close method for the initialized client', done => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      client.initialize().catch(err => {
        throw err;
      });
      assert(client.spannerStub);
      client
        .close()
        .then(() => {
          done();
        })
        .catch(err => {
          throw err;
        });
    });

    it('has close method for the non-initialized client', done => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      assert.strictEqual(client.spannerStub, undefined);
      client
        .close()
        .then(() => {
          done();
        })
        .catch(err => {
          throw err;
        });
    });

    it('has getProjectId method', async () => {
      const fakeProjectId = 'fake-project-id';
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      client.auth.getProjectId = sinon.stub().resolves(fakeProjectId);
      const result = await client.getProjectId();
      assert.strictEqual(result, fakeProjectId);
      assert((client.auth.getProjectId as SinonStub).calledWithExactly());
    });

    it('has getProjectId method with callback', async () => {
      const fakeProjectId = 'fake-project-id';
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      client.auth.getProjectId = sinon
        .stub()
        .callsArgWith(0, null, fakeProjectId);
      const promise = new Promise((resolve, reject) => {
        client.getProjectId((err?: Error | null, projectId?: string | null) => {
          if (err) {
            reject(err);
          } else {
            resolve(projectId);
          }
        });
      });
      const result = await promise;
      assert.strictEqual(result, fakeProjectId);
    });
  });

  describe('createSession', () => {
    it('invokes createSession without error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.CreateSessionRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.CreateSessionRequest',
        ['database'],
      );
      request.database = defaultValue1;
      const expectedHeaderRequestParams = `database=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.Session(),
      );
      client.innerApiCalls.createSession = stubSimpleCall(expectedResponse);
      const [response] = await client.createSession(request);
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.createSession as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.createSession as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes createSession without error using callback', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.CreateSessionRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.CreateSessionRequest',
        ['database'],
      );
      request.database = defaultValue1;
      const expectedHeaderRequestParams = `database=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.Session(),
      );
      client.innerApiCalls.createSession =
        stubSimpleCallWithCallback(expectedResponse);
      const promise = new Promise((resolve, reject) => {
        client.createSession(
          request,
          (
            err?: Error | null,
            result?: protos.google.spanner.v1.ISession | null,
          ) => {
            if (err) {
              reject(err);
            } else {
              resolve(result);
            }
          },
        );
      });
      const response = await promise;
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.createSession as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.createSession as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes createSession with error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.CreateSessionRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.CreateSessionRequest',
        ['database'],
      );
      request.database = defaultValue1;
      const expectedHeaderRequestParams = `database=${defaultValue1 ?? ''}`;
      const expectedError = new Error('expected');
      client.innerApiCalls.createSession = stubSimpleCall(
        undefined,
        expectedError,
      );
      await assert.rejects(client.createSession(request), expectedError);
      const actualRequest = (
        client.innerApiCalls.createSession as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.createSession as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes createSession with closed client', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.CreateSessionRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.CreateSessionRequest',
        ['database'],
      );
      request.database = defaultValue1;
      const expectedError = new Error('The client has already been closed.');
      client.close().catch(err => {
        throw err;
      });
      await assert.rejects(client.createSession(request), expectedError);
    });
  });

  describe('batchCreateSessions', () => {
    it('invokes batchCreateSessions without error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.BatchCreateSessionsRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.BatchCreateSessionsRequest',
        ['database'],
      );
      request.database = defaultValue1;
      const expectedHeaderRequestParams = `database=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.BatchCreateSessionsResponse(),
      );
      client.innerApiCalls.batchCreateSessions =
        stubSimpleCall(expectedResponse);
      const [response] = await client.batchCreateSessions(request);
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.batchCreateSessions as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.batchCreateSessions as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes batchCreateSessions without error using callback', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.BatchCreateSessionsRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.BatchCreateSessionsRequest',
        ['database'],
      );
      request.database = defaultValue1;
      const expectedHeaderRequestParams = `database=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.BatchCreateSessionsResponse(),
      );
      client.innerApiCalls.batchCreateSessions =
        stubSimpleCallWithCallback(expectedResponse);
      const promise = new Promise((resolve, reject) => {
        client.batchCreateSessions(
          request,
          (
            err?: Error | null,
            result?: protos.google.spanner.v1.IBatchCreateSessionsResponse | null,
          ) => {
            if (err) {
              reject(err);
            } else {
              resolve(result);
            }
          },
        );
      });
      const response = await promise;
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.batchCreateSessions as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.batchCreateSessions as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes batchCreateSessions with error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.BatchCreateSessionsRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.BatchCreateSessionsRequest',
        ['database'],
      );
      request.database = defaultValue1;
      const expectedHeaderRequestParams = `database=${defaultValue1 ?? ''}`;
      const expectedError = new Error('expected');
      client.innerApiCalls.batchCreateSessions = stubSimpleCall(
        undefined,
        expectedError,
      );
      await assert.rejects(client.batchCreateSessions(request), expectedError);
      const actualRequest = (
        client.innerApiCalls.batchCreateSessions as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.batchCreateSessions as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes batchCreateSessions with closed client', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.BatchCreateSessionsRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.BatchCreateSessionsRequest',
        ['database'],
      );
      request.database = defaultValue1;
      const expectedError = new Error('The client has already been closed.');
      client.close().catch(err => {
        throw err;
      });
      await assert.rejects(client.batchCreateSessions(request), expectedError);
    });
  });

  describe('getSession', () => {
    it('invokes getSession without error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.GetSessionRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.GetSessionRequest',
        ['name'],
      );
      request.name = defaultValue1;
      const expectedHeaderRequestParams = `name=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.Session(),
      );
      client.innerApiCalls.getSession = stubSimpleCall(expectedResponse);
      const [response] = await client.getSession(request);
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.getSession as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.getSession as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes getSession without error using callback', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.GetSessionRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.GetSessionRequest',
        ['name'],
      );
      request.name = defaultValue1;
      const expectedHeaderRequestParams = `name=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.Session(),
      );
      client.innerApiCalls.getSession =
        stubSimpleCallWithCallback(expectedResponse);
      const promise = new Promise((resolve, reject) => {
        client.getSession(
          request,
          (
            err?: Error | null,
            result?: protos.google.spanner.v1.ISession | null,
          ) => {
            if (err) {
              reject(err);
            } else {
              resolve(result);
            }
          },
        );
      });
      const response = await promise;
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.getSession as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.getSession as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes getSession with error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.GetSessionRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.GetSessionRequest',
        ['name'],
      );
      request.name = defaultValue1;
      const expectedHeaderRequestParams = `name=${defaultValue1 ?? ''}`;
      const expectedError = new Error('expected');
      client.innerApiCalls.getSession = stubSimpleCall(
        undefined,
        expectedError,
      );
      await assert.rejects(client.getSession(request), expectedError);
      const actualRequest = (
        client.innerApiCalls.getSession as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.getSession as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes getSession with closed client', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.GetSessionRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.GetSessionRequest',
        ['name'],
      );
      request.name = defaultValue1;
      const expectedError = new Error('The client has already been closed.');
      client.close().catch(err => {
        throw err;
      });
      await assert.rejects(client.getSession(request), expectedError);
    });
  });

  describe('deleteSession', () => {
    it('invokes deleteSession without error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.DeleteSessionRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.DeleteSessionRequest',
        ['name'],
      );
      request.name = defaultValue1;
      const expectedHeaderRequestParams = `name=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.protobuf.Empty(),
      );
      client.innerApiCalls.deleteSession = stubSimpleCall(expectedResponse);
      const [response] = await client.deleteSession(request);
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.deleteSession as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.deleteSession as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes deleteSession without error using callback', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.DeleteSessionRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.DeleteSessionRequest',
        ['name'],
      );
      request.name = defaultValue1;
      const expectedHeaderRequestParams = `name=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.protobuf.Empty(),
      );
      client.innerApiCalls.deleteSession =
        stubSimpleCallWithCallback(expectedResponse);
      const promise = new Promise((resolve, reject) => {
        client.deleteSession(
          request,
          (
            err?: Error | null,
            result?: protos.google.protobuf.IEmpty | null,
          ) => {
            if (err) {
              reject(err);
            } else {
              resolve(result);
            }
          },
        );
      });
      const response = await promise;
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.deleteSession as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.deleteSession as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes deleteSession with error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.DeleteSessionRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.DeleteSessionRequest',
        ['name'],
      );
      request.name = defaultValue1;
      const expectedHeaderRequestParams = `name=${defaultValue1 ?? ''}`;
      const expectedError = new Error('expected');
      client.innerApiCalls.deleteSession = stubSimpleCall(
        undefined,
        expectedError,
      );
      await assert.rejects(client.deleteSession(request), expectedError);
      const actualRequest = (
        client.innerApiCalls.deleteSession as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.deleteSession as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes deleteSession with closed client', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.DeleteSessionRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.DeleteSessionRequest',
        ['name'],
      );
      request.name = defaultValue1;
      const expectedError = new Error('The client has already been closed.');
      client.close().catch(err => {
        throw err;
      });
      await assert.rejects(client.deleteSession(request), expectedError);
    });
  });

  describe('executeSql', () => {
    it('invokes executeSql without error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ExecuteSqlRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ExecuteSqlRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.ResultSet(),
      );
      client.innerApiCalls.executeSql = stubSimpleCall(expectedResponse);
      const [response] = await client.executeSql(request);
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.executeSql as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.executeSql as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes executeSql without error using callback', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ExecuteSqlRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ExecuteSqlRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.ResultSet(),
      );
      client.innerApiCalls.executeSql =
        stubSimpleCallWithCallback(expectedResponse);
      const promise = new Promise((resolve, reject) => {
        client.executeSql(
          request,
          (
            err?: Error | null,
            result?: protos.google.spanner.v1.IResultSet | null,
          ) => {
            if (err) {
              reject(err);
            } else {
              resolve(result);
            }
          },
        );
      });
      const response = await promise;
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.executeSql as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.executeSql as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes executeSql with error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ExecuteSqlRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ExecuteSqlRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedError = new Error('expected');
      client.innerApiCalls.executeSql = stubSimpleCall(
        undefined,
        expectedError,
      );
      await assert.rejects(client.executeSql(request), expectedError);
      const actualRequest = (
        client.innerApiCalls.executeSql as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.executeSql as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes executeSql with closed client', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ExecuteSqlRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ExecuteSqlRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedError = new Error('The client has already been closed.');
      client.close().catch(err => {
        throw err;
      });
      await assert.rejects(client.executeSql(request), expectedError);
    });
  });

  describe('executeBatchDml', () => {
    it('invokes executeBatchDml without error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ExecuteBatchDmlRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ExecuteBatchDmlRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.ExecuteBatchDmlResponse(),
      );
      client.innerApiCalls.executeBatchDml = stubSimpleCall(expectedResponse);
      const [response] = await client.executeBatchDml(request);
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.executeBatchDml as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.executeBatchDml as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes executeBatchDml without error using callback', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ExecuteBatchDmlRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ExecuteBatchDmlRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.ExecuteBatchDmlResponse(),
      );
      client.innerApiCalls.executeBatchDml =
        stubSimpleCallWithCallback(expectedResponse);
      const promise = new Promise((resolve, reject) => {
        client.executeBatchDml(
          request,
          (
            err?: Error | null,
            result?: protos.google.spanner.v1.IExecuteBatchDmlResponse | null,
          ) => {
            if (err) {
              reject(err);
            } else {
              resolve(result);
            }
          },
        );
      });
      const response = await promise;
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.executeBatchDml as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.executeBatchDml as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes executeBatchDml with error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ExecuteBatchDmlRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ExecuteBatchDmlRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedError = new Error('expected');
      client.innerApiCalls.executeBatchDml = stubSimpleCall(
        undefined,
        expectedError,
      );
      await assert.rejects(client.executeBatchDml(request), expectedError);
      const actualRequest = (
        client.innerApiCalls.executeBatchDml as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.executeBatchDml as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes executeBatchDml with closed client', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ExecuteBatchDmlRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ExecuteBatchDmlRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedError = new Error('The client has already been closed.');
      client.close().catch(err => {
        throw err;
      });
      await assert.rejects(client.executeBatchDml(request), expectedError);
    });
  });

  describe('read', () => {
    it('invokes read without error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ReadRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ReadRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.ResultSet(),
      );
      client.innerApiCalls.read = stubSimpleCall(expectedResponse);
      const [response] = await client.read(request);
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (client.innerApiCalls.read as SinonStub).getCall(0)
        .args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.read as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes read without error using callback', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ReadRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ReadRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.ResultSet(),
      );
      client.innerApiCalls.read = stubSimpleCallWithCallback(expectedResponse);
      const promise = new Promise((resolve, reject) => {
        client.read(
          request,
          (
            err?: Error | null,
            result?: protos.google.spanner.v1.IResultSet | null,
          ) => {
            if (err) {
              reject(err);
            } else {
              resolve(result);
            }
          },
        );
      });
      const response = await promise;
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (client.innerApiCalls.read as SinonStub).getCall(0)
        .args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.read as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes read with error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ReadRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ReadRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedError = new Error('expected');
      client.innerApiCalls.read = stubSimpleCall(undefined, expectedError);
      await assert.rejects(client.read(request), expectedError);
      const actualRequest = (client.innerApiCalls.read as SinonStub).getCall(0)
        .args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.read as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes read with closed client', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ReadRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ReadRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedError = new Error('The client has already been closed.');
      client.close().catch(err => {
        throw err;
      });
      await assert.rejects(client.read(request), expectedError);
    });
  });

  describe('beginTransaction', () => {
    it('invokes beginTransaction without error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.BeginTransactionRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.BeginTransactionRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.Transaction(),
      );
      client.innerApiCalls.beginTransaction = stubSimpleCall(expectedResponse);
      const [response] = await client.beginTransaction(request);
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.beginTransaction as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.beginTransaction as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes beginTransaction without error using callback', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.BeginTransactionRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.BeginTransactionRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.Transaction(),
      );
      client.innerApiCalls.beginTransaction =
        stubSimpleCallWithCallback(expectedResponse);
      const promise = new Promise((resolve, reject) => {
        client.beginTransaction(
          request,
          (
            err?: Error | null,
            result?: protos.google.spanner.v1.ITransaction | null,
          ) => {
            if (err) {
              reject(err);
            } else {
              resolve(result);
            }
          },
        );
      });
      const response = await promise;
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.beginTransaction as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.beginTransaction as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes beginTransaction with error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.BeginTransactionRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.BeginTransactionRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedError = new Error('expected');
      client.innerApiCalls.beginTransaction = stubSimpleCall(
        undefined,
        expectedError,
      );
      await assert.rejects(client.beginTransaction(request), expectedError);
      const actualRequest = (
        client.innerApiCalls.beginTransaction as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.beginTransaction as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes beginTransaction with closed client', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.BeginTransactionRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.BeginTransactionRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedError = new Error('The client has already been closed.');
      client.close().catch(err => {
        throw err;
      });
      await assert.rejects(client.beginTransaction(request), expectedError);
    });
  });

  describe('commit', () => {
    it('invokes commit without error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.CommitRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.CommitRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.CommitResponse(),
      );
      client.innerApiCalls.commit = stubSimpleCall(expectedResponse);
      const [response] = await client.commit(request);
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (client.innerApiCalls.commit as SinonStub).getCall(
        0,
      ).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.commit as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes commit without error using callback', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.CommitRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.CommitRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.CommitResponse(),
      );
      client.innerApiCalls.commit =
        stubSimpleCallWithCallback(expectedResponse);
      const promise = new Promise((resolve, reject) => {
        client.commit(
          request,
          (
            err?: Error | null,
            result?: protos.google.spanner.v1.ICommitResponse | null,
          ) => {
            if (err) {
              reject(err);
            } else {
              resolve(result);
            }
          },
        );
      });
      const response = await promise;
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (client.innerApiCalls.commit as SinonStub).getCall(
        0,
      ).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.commit as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes commit with error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.CommitRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.CommitRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedError = new Error('expected');
      client.innerApiCalls.commit = stubSimpleCall(undefined, expectedError);
      await assert.rejects(client.commit(request), expectedError);
      const actualRequest = (client.innerApiCalls.commit as SinonStub).getCall(
        0,
      ).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.commit as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes commit with closed client', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.CommitRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.CommitRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedError = new Error('The client has already been closed.');
      client.close().catch(err => {
        throw err;
      });
      await assert.rejects(client.commit(request), expectedError);
    });
  });

  describe('rollback', () => {
    it('invokes rollback without error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.RollbackRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.RollbackRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.protobuf.Empty(),
      );
      client.innerApiCalls.rollback = stubSimpleCall(expectedResponse);
      const [response] = await client.rollback(request);
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.rollback as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.rollback as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes rollback without error using callback', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.RollbackRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.RollbackRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.protobuf.Empty(),
      );
      client.innerApiCalls.rollback =
        stubSimpleCallWithCallback(expectedResponse);
      const promise = new Promise((resolve, reject) => {
        client.rollback(
          request,
          (
            err?: Error | null,
            result?: protos.google.protobuf.IEmpty | null,
          ) => {
            if (err) {
              reject(err);
            } else {
              resolve(result);
            }
          },
        );
      });
      const response = await promise;
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.rollback as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.rollback as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes rollback with error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.RollbackRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.RollbackRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedError = new Error('expected');
      client.innerApiCalls.rollback = stubSimpleCall(undefined, expectedError);
      await assert.rejects(client.rollback(request), expectedError);
      const actualRequest = (
        client.innerApiCalls.rollback as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.rollback as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes rollback with closed client', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.RollbackRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.RollbackRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedError = new Error('The client has already been closed.');
      client.close().catch(err => {
        throw err;
      });
      await assert.rejects(client.rollback(request), expectedError);
    });
  });

  describe('partitionQuery', () => {
    it('invokes partitionQuery without error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.PartitionQueryRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.PartitionQueryRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.PartitionResponse(),
      );
      client.innerApiCalls.partitionQuery = stubSimpleCall(expectedResponse);
      const [response] = await client.partitionQuery(request);
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.partitionQuery as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.partitionQuery as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes partitionQuery without error using callback', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.PartitionQueryRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.PartitionQueryRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.PartitionResponse(),
      );
      client.innerApiCalls.partitionQuery =
        stubSimpleCallWithCallback(expectedResponse);
      const promise = new Promise((resolve, reject) => {
        client.partitionQuery(
          request,
          (
            err?: Error | null,
            result?: protos.google.spanner.v1.IPartitionResponse | null,
          ) => {
            if (err) {
              reject(err);
            } else {
              resolve(result);
            }
          },
        );
      });
      const response = await promise;
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.partitionQuery as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.partitionQuery as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes partitionQuery with error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.PartitionQueryRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.PartitionQueryRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedError = new Error('expected');
      client.innerApiCalls.partitionQuery = stubSimpleCall(
        undefined,
        expectedError,
      );
      await assert.rejects(client.partitionQuery(request), expectedError);
      const actualRequest = (
        client.innerApiCalls.partitionQuery as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.partitionQuery as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes partitionQuery with closed client', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.PartitionQueryRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.PartitionQueryRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedError = new Error('The client has already been closed.');
      client.close().catch(err => {
        throw err;
      });
      await assert.rejects(client.partitionQuery(request), expectedError);
    });
  });

  describe('partitionRead', () => {
    it('invokes partitionRead without error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.PartitionReadRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.PartitionReadRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.PartitionResponse(),
      );
      client.innerApiCalls.partitionRead = stubSimpleCall(expectedResponse);
      const [response] = await client.partitionRead(request);
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.partitionRead as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.partitionRead as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes partitionRead without error using callback', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.PartitionReadRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.PartitionReadRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.PartitionResponse(),
      );
      client.innerApiCalls.partitionRead =
        stubSimpleCallWithCallback(expectedResponse);
      const promise = new Promise((resolve, reject) => {
        client.partitionRead(
          request,
          (
            err?: Error | null,
            result?: protos.google.spanner.v1.IPartitionResponse | null,
          ) => {
            if (err) {
              reject(err);
            } else {
              resolve(result);
            }
          },
        );
      });
      const response = await promise;
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.partitionRead as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.partitionRead as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes partitionRead with error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.PartitionReadRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.PartitionReadRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedError = new Error('expected');
      client.innerApiCalls.partitionRead = stubSimpleCall(
        undefined,
        expectedError,
      );
      await assert.rejects(client.partitionRead(request), expectedError);
      const actualRequest = (
        client.innerApiCalls.partitionRead as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.partitionRead as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes partitionRead with closed client', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.PartitionReadRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.PartitionReadRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedError = new Error('The client has already been closed.');
      client.close().catch(err => {
        throw err;
      });
      await assert.rejects(client.partitionRead(request), expectedError);
    });
  });

  describe('executeStreamingSql', () => {
    it('invokes executeStreamingSql without error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ExecuteSqlRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ExecuteSqlRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.PartialResultSet(),
      );
      client.innerApiCalls.executeStreamingSql =
        stubServerStreamingCall(expectedResponse);
      const stream = client.executeStreamingSql(request);
      const promise = new Promise((resolve, reject) => {
        stream.on(
          'data',
          (response: protos.google.spanner.v1.PartialResultSet) => {
            resolve(response);
          },
        );
        stream.on('error', (err: Error) => {
          reject(err);
        });
      });
      const response = await promise;
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.executeStreamingSql as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.executeStreamingSql as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes executeStreamingSql without error and gaxServerStreamingRetries enabled', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
        gaxServerStreamingRetries: true,
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ExecuteSqlRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ExecuteSqlRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.PartialResultSet(),
      );
      client.innerApiCalls.executeStreamingSql =
        stubServerStreamingCall(expectedResponse);
      const stream = client.executeStreamingSql(request);
      const promise = new Promise((resolve, reject) => {
        stream.on(
          'data',
          (response: protos.google.spanner.v1.PartialResultSet) => {
            resolve(response);
          },
        );
        stream.on('error', (err: Error) => {
          reject(err);
        });
      });
      const response = await promise;
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.executeStreamingSql as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.executeStreamingSql as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes executeStreamingSql with error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ExecuteSqlRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ExecuteSqlRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedError = new Error('expected');
      client.innerApiCalls.executeStreamingSql = stubServerStreamingCall(
        undefined,
        expectedError,
      );
      const stream = client.executeStreamingSql(request);
      const promise = new Promise((resolve, reject) => {
        stream.on(
          'data',
          (response: protos.google.spanner.v1.PartialResultSet) => {
            resolve(response);
          },
        );
        stream.on('error', (err: Error) => {
          reject(err);
        });
      });
      await assert.rejects(promise, expectedError);
      const actualRequest = (
        client.innerApiCalls.executeStreamingSql as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.executeStreamingSql as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes executeStreamingSql with closed client', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ExecuteSqlRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ExecuteSqlRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedError = new Error('The client has already been closed.');
      client.close().catch(err => {
        throw err;
      });
      const stream = client.executeStreamingSql(request, {
        retryRequestOptions: {noResponseRetries: 0},
      });
      const promise = new Promise((resolve, reject) => {
        stream.on(
          'data',
          (response: protos.google.spanner.v1.PartialResultSet) => {
            resolve(response);
          },
        );
        stream.on('error', (err: Error) => {
          reject(err);
        });
      });
      await assert.rejects(promise, expectedError);
    });
    it('should create a client with gaxServerStreamingRetries enabled', () => {
      const client = new spannerModule.v1.SpannerClient({
        gaxServerStreamingRetries: true,
      });
      assert(client);
    });
  });

  describe('streamingRead', () => {
    it('invokes streamingRead without error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ReadRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ReadRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.PartialResultSet(),
      );
      client.innerApiCalls.streamingRead =
        stubServerStreamingCall(expectedResponse);
      const stream = client.streamingRead(request);
      const promise = new Promise((resolve, reject) => {
        stream.on(
          'data',
          (response: protos.google.spanner.v1.PartialResultSet) => {
            resolve(response);
          },
        );
        stream.on('error', (err: Error) => {
          reject(err);
        });
      });
      const response = await promise;
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.streamingRead as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.streamingRead as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes streamingRead without error and gaxServerStreamingRetries enabled', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
        gaxServerStreamingRetries: true,
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ReadRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ReadRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.PartialResultSet(),
      );
      client.innerApiCalls.streamingRead =
        stubServerStreamingCall(expectedResponse);
      const stream = client.streamingRead(request);
      const promise = new Promise((resolve, reject) => {
        stream.on(
          'data',
          (response: protos.google.spanner.v1.PartialResultSet) => {
            resolve(response);
          },
        );
        stream.on('error', (err: Error) => {
          reject(err);
        });
      });
      const response = await promise;
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.streamingRead as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.streamingRead as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes streamingRead with error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ReadRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ReadRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedError = new Error('expected');
      client.innerApiCalls.streamingRead = stubServerStreamingCall(
        undefined,
        expectedError,
      );
      const stream = client.streamingRead(request);
      const promise = new Promise((resolve, reject) => {
        stream.on(
          'data',
          (response: protos.google.spanner.v1.PartialResultSet) => {
            resolve(response);
          },
        );
        stream.on('error', (err: Error) => {
          reject(err);
        });
      });
      await assert.rejects(promise, expectedError);
      const actualRequest = (
        client.innerApiCalls.streamingRead as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.streamingRead as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes streamingRead with closed client', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ReadRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ReadRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedError = new Error('The client has already been closed.');
      client.close().catch(err => {
        throw err;
      });
      const stream = client.streamingRead(request, {
        retryRequestOptions: {noResponseRetries: 0},
      });
      const promise = new Promise((resolve, reject) => {
        stream.on(
          'data',
          (response: protos.google.spanner.v1.PartialResultSet) => {
            resolve(response);
          },
        );
        stream.on('error', (err: Error) => {
          reject(err);
        });
      });
      await assert.rejects(promise, expectedError);
    });
    it('should create a client with gaxServerStreamingRetries enabled', () => {
      const client = new spannerModule.v1.SpannerClient({
        gaxServerStreamingRetries: true,
      });
      assert(client);
    });
  });

  describe('batchWrite', () => {
    it('invokes batchWrite without error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.BatchWriteRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.BatchWriteRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.BatchWriteResponse(),
      );
      client.innerApiCalls.batchWrite =
        stubServerStreamingCall(expectedResponse);
      const stream = client.batchWrite(request);
      const promise = new Promise((resolve, reject) => {
        stream.on(
          'data',
          (response: protos.google.spanner.v1.BatchWriteResponse) => {
            resolve(response);
          },
        );
        stream.on('error', (err: Error) => {
          reject(err);
        });
      });
      const response = await promise;
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.batchWrite as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.batchWrite as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes batchWrite without error and gaxServerStreamingRetries enabled', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
        gaxServerStreamingRetries: true,
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.BatchWriteRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.BatchWriteRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedResponse = generateSampleMessage(
        new protos.google.spanner.v1.BatchWriteResponse(),
      );
      client.innerApiCalls.batchWrite =
        stubServerStreamingCall(expectedResponse);
      const stream = client.batchWrite(request);
      const promise = new Promise((resolve, reject) => {
        stream.on(
          'data',
          (response: protos.google.spanner.v1.BatchWriteResponse) => {
            resolve(response);
          },
        );
        stream.on('error', (err: Error) => {
          reject(err);
        });
      });
      const response = await promise;
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.batchWrite as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.batchWrite as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes batchWrite with error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.BatchWriteRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.BatchWriteRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedHeaderRequestParams = `session=${defaultValue1 ?? ''}`;
      const expectedError = new Error('expected');
      client.innerApiCalls.batchWrite = stubServerStreamingCall(
        undefined,
        expectedError,
      );
      const stream = client.batchWrite(request);
      const promise = new Promise((resolve, reject) => {
        stream.on(
          'data',
          (response: protos.google.spanner.v1.BatchWriteResponse) => {
            resolve(response);
          },
        );
        stream.on('error', (err: Error) => {
          reject(err);
        });
      });
      await assert.rejects(promise, expectedError);
      const actualRequest = (
        client.innerApiCalls.batchWrite as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.batchWrite as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes batchWrite with closed client', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.BatchWriteRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.BatchWriteRequest',
        ['session'],
      );
      request.session = defaultValue1;
      const expectedError = new Error('The client has already been closed.');
      client.close().catch(err => {
        throw err;
      });
      const stream = client.batchWrite(request, {
        retryRequestOptions: {noResponseRetries: 0},
      });
      const promise = new Promise((resolve, reject) => {
        stream.on(
          'data',
          (response: protos.google.spanner.v1.BatchWriteResponse) => {
            resolve(response);
          },
        );
        stream.on('error', (err: Error) => {
          reject(err);
        });
      });
      await assert.rejects(promise, expectedError);
    });
    it('should create a client with gaxServerStreamingRetries enabled', () => {
      const client = new spannerModule.v1.SpannerClient({
        gaxServerStreamingRetries: true,
      });
      assert(client);
    });
  });

  describe('listSessions', () => {
    it('invokes listSessions without error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ListSessionsRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ListSessionsRequest',
        ['database'],
      );
      request.database = defaultValue1;
      const expectedHeaderRequestParams = `database=${defaultValue1 ?? ''}`;
      const expectedResponse = [
        generateSampleMessage(new protos.google.spanner.v1.Session()),
        generateSampleMessage(new protos.google.spanner.v1.Session()),
        generateSampleMessage(new protos.google.spanner.v1.Session()),
      ];
      client.innerApiCalls.listSessions = stubSimpleCall(expectedResponse);
      const [response] = await client.listSessions(request);
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.listSessions as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.listSessions as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes listSessions without error using callback', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ListSessionsRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ListSessionsRequest',
        ['database'],
      );
      request.database = defaultValue1;
      const expectedHeaderRequestParams = `database=${defaultValue1 ?? ''}`;
      const expectedResponse = [
        generateSampleMessage(new protos.google.spanner.v1.Session()),
        generateSampleMessage(new protos.google.spanner.v1.Session()),
        generateSampleMessage(new protos.google.spanner.v1.Session()),
      ];
      client.innerApiCalls.listSessions =
        stubSimpleCallWithCallback(expectedResponse);
      const promise = new Promise((resolve, reject) => {
        client.listSessions(
          request,
          (
            err?: Error | null,
            result?: protos.google.spanner.v1.ISession[] | null,
          ) => {
            if (err) {
              reject(err);
            } else {
              resolve(result);
            }
          },
        );
      });
      const response = await promise;
      assert.deepStrictEqual(response, expectedResponse);
      const actualRequest = (
        client.innerApiCalls.listSessions as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.listSessions as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes listSessions with error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ListSessionsRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ListSessionsRequest',
        ['database'],
      );
      request.database = defaultValue1;
      const expectedHeaderRequestParams = `database=${defaultValue1 ?? ''}`;
      const expectedError = new Error('expected');
      client.innerApiCalls.listSessions = stubSimpleCall(
        undefined,
        expectedError,
      );
      await assert.rejects(client.listSessions(request), expectedError);
      const actualRequest = (
        client.innerApiCalls.listSessions as SinonStub
      ).getCall(0).args[0];
      assert.deepStrictEqual(actualRequest, request);
      const actualHeaderRequestParams = (
        client.innerApiCalls.listSessions as SinonStub
      ).getCall(0).args[1].otherArgs.headers['x-goog-request-params'];
      assert(actualHeaderRequestParams.includes(expectedHeaderRequestParams));
    });

    it('invokes listSessionsStream without error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ListSessionsRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ListSessionsRequest',
        ['database'],
      );
      request.database = defaultValue1;
      const expectedHeaderRequestParams = `database=${defaultValue1 ?? ''}`;
      const expectedResponse = [
        generateSampleMessage(new protos.google.spanner.v1.Session()),
        generateSampleMessage(new protos.google.spanner.v1.Session()),
        generateSampleMessage(new protos.google.spanner.v1.Session()),
      ];
      client.descriptors.page.listSessions.createStream =
        stubPageStreamingCall(expectedResponse);
      const stream = client.listSessionsStream(request);
      const promise = new Promise((resolve, reject) => {
        const responses: protos.google.spanner.v1.Session[] = [];
        stream.on('data', (response: protos.google.spanner.v1.Session) => {
          responses.push(response);
        });
        stream.on('end', () => {
          resolve(responses);
        });
        stream.on('error', (err: Error) => {
          reject(err);
        });
      });
      const responses = await promise;
      assert.deepStrictEqual(responses, expectedResponse);
      assert(
        (client.descriptors.page.listSessions.createStream as SinonStub)
          .getCall(0)
          .calledWith(client.innerApiCalls.listSessions, request),
      );
      assert(
        (client.descriptors.page.listSessions.createStream as SinonStub)
          .getCall(0)
          .args[2].otherArgs.headers[
            'x-goog-request-params'
          ].includes(expectedHeaderRequestParams),
      );
    });

    it('invokes listSessionsStream with error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ListSessionsRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ListSessionsRequest',
        ['database'],
      );
      request.database = defaultValue1;
      const expectedHeaderRequestParams = `database=${defaultValue1 ?? ''}`;
      const expectedError = new Error('expected');
      client.descriptors.page.listSessions.createStream = stubPageStreamingCall(
        undefined,
        expectedError,
      );
      const stream = client.listSessionsStream(request);
      const promise = new Promise((resolve, reject) => {
        const responses: protos.google.spanner.v1.Session[] = [];
        stream.on('data', (response: protos.google.spanner.v1.Session) => {
          responses.push(response);
        });
        stream.on('end', () => {
          resolve(responses);
        });
        stream.on('error', (err: Error) => {
          reject(err);
        });
      });
      await assert.rejects(promise, expectedError);
      assert(
        (client.descriptors.page.listSessions.createStream as SinonStub)
          .getCall(0)
          .calledWith(client.innerApiCalls.listSessions, request),
      );
      assert(
        (client.descriptors.page.listSessions.createStream as SinonStub)
          .getCall(0)
          .args[2].otherArgs.headers[
            'x-goog-request-params'
          ].includes(expectedHeaderRequestParams),
      );
    });

    it('uses async iteration with listSessions without error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ListSessionsRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ListSessionsRequest',
        ['database'],
      );
      request.database = defaultValue1;
      const expectedHeaderRequestParams = `database=${defaultValue1 ?? ''}`;
      const expectedResponse = [
        generateSampleMessage(new protos.google.spanner.v1.Session()),
        generateSampleMessage(new protos.google.spanner.v1.Session()),
        generateSampleMessage(new protos.google.spanner.v1.Session()),
      ];
      client.descriptors.page.listSessions.asyncIterate =
        stubAsyncIterationCall(expectedResponse);
      const responses: protos.google.spanner.v1.ISession[] = [];
      const iterable = client.listSessionsAsync(request);
      for await (const resource of iterable) {
        responses.push(resource!);
      }
      assert.deepStrictEqual(responses, expectedResponse);
      assert.deepStrictEqual(
        (
          client.descriptors.page.listSessions.asyncIterate as SinonStub
        ).getCall(0).args[1],
        request,
      );
      assert(
        (client.descriptors.page.listSessions.asyncIterate as SinonStub)
          .getCall(0)
          .args[2].otherArgs.headers[
            'x-goog-request-params'
          ].includes(expectedHeaderRequestParams),
      );
    });

    it('uses async iteration with listSessions with error', async () => {
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      const request = generateSampleMessage(
        new protos.google.spanner.v1.ListSessionsRequest(),
      );
      const defaultValue1 = getTypeDefaultValue(
        '.google.spanner.v1.ListSessionsRequest',
        ['database'],
      );
      request.database = defaultValue1;
      const expectedHeaderRequestParams = `database=${defaultValue1 ?? ''}`;
      const expectedError = new Error('expected');
      client.descriptors.page.listSessions.asyncIterate =
        stubAsyncIterationCall(undefined, expectedError);
      const iterable = client.listSessionsAsync(request);
      await assert.rejects(async () => {
        const responses: protos.google.spanner.v1.ISession[] = [];
        for await (const resource of iterable) {
          responses.push(resource!);
        }
      });
      assert.deepStrictEqual(
        (
          client.descriptors.page.listSessions.asyncIterate as SinonStub
        ).getCall(0).args[1],
        request,
      );
      assert(
        (client.descriptors.page.listSessions.asyncIterate as SinonStub)
          .getCall(0)
          .args[2].otherArgs.headers[
            'x-goog-request-params'
          ].includes(expectedHeaderRequestParams),
      );
    });
  });

  describe('Path templates', () => {
    describe('database', async () => {
      const fakePath = '/rendered/path/database';
      const expectedParameters = {
        project: 'projectValue',
        instance: 'instanceValue',
        database: 'databaseValue',
      };
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      client.pathTemplates.databasePathTemplate.render = sinon
        .stub()
        .returns(fakePath);
      client.pathTemplates.databasePathTemplate.match = sinon
        .stub()
        .returns(expectedParameters);

      it('databasePath', () => {
        const result = client.databasePath(
          'projectValue',
          'instanceValue',
          'databaseValue',
        );
        assert.strictEqual(result, fakePath);
        assert(
          (client.pathTemplates.databasePathTemplate.render as SinonStub)
            .getCall(-1)
            .calledWith(expectedParameters),
        );
      });

      it('matchProjectFromDatabaseName', () => {
        const result = client.matchProjectFromDatabaseName(fakePath);
        assert.strictEqual(result, 'projectValue');
        assert(
          (client.pathTemplates.databasePathTemplate.match as SinonStub)
            .getCall(-1)
            .calledWith(fakePath),
        );
      });

      it('matchInstanceFromDatabaseName', () => {
        const result = client.matchInstanceFromDatabaseName(fakePath);
        assert.strictEqual(result, 'instanceValue');
        assert(
          (client.pathTemplates.databasePathTemplate.match as SinonStub)
            .getCall(-1)
            .calledWith(fakePath),
        );
      });

      it('matchDatabaseFromDatabaseName', () => {
        const result = client.matchDatabaseFromDatabaseName(fakePath);
        assert.strictEqual(result, 'databaseValue');
        assert(
          (client.pathTemplates.databasePathTemplate.match as SinonStub)
            .getCall(-1)
            .calledWith(fakePath),
        );
      });
    });

    describe('session', async () => {
      const fakePath = '/rendered/path/session';
      const expectedParameters = {
        project: 'projectValue',
        instance: 'instanceValue',
        database: 'databaseValue',
        session: 'sessionValue',
      };
      const client = new spannerModule.v1.SpannerClient({
        credentials: {client_email: 'bogus', private_key: 'bogus'},
        projectId: 'bogus',
      });
      await client.initialize();
      client.pathTemplates.sessionPathTemplate.render = sinon
        .stub()
        .returns(fakePath);
      client.pathTemplates.sessionPathTemplate.match = sinon
        .stub()
        .returns(expectedParameters);

      it('sessionPath', () => {
        const result = client.sessionPath(
          'projectValue',
          'instanceValue',
          'databaseValue',
          'sessionValue',
        );
        assert.strictEqual(result, fakePath);
        assert(
          (client.pathTemplates.sessionPathTemplate.render as SinonStub)
            .getCall(-1)
            .calledWith(expectedParameters),
        );
      });

      it('matchProjectFromSessionName', () => {
        const result = client.matchProjectFromSessionName(fakePath);
        assert.strictEqual(result, 'projectValue');
        assert(
          (client.pathTemplates.sessionPathTemplate.match as SinonStub)
            .getCall(-1)
            .calledWith(fakePath),
        );
      });

      it('matchInstanceFromSessionName', () => {
        const result = client.matchInstanceFromSessionName(fakePath);
        assert.strictEqual(result, 'instanceValue');
        assert(
          (client.pathTemplates.sessionPathTemplate.match as SinonStub)
            .getCall(-1)
            .calledWith(fakePath),
        );
      });

      it('matchDatabaseFromSessionName', () => {
        const result = client.matchDatabaseFromSessionName(fakePath);
        assert.strictEqual(result, 'databaseValue');
        assert(
          (client.pathTemplates.sessionPathTemplate.match as SinonStub)
            .getCall(-1)
            .calledWith(fakePath),
        );
      });

      it('matchSessionFromSessionName', () => {
        const result = client.matchSessionFromSessionName(fakePath);
        assert.strictEqual(result, 'sessionValue');
        assert(
          (client.pathTemplates.sessionPathTemplate.match as SinonStub)
            .getCall(-1)
            .calledWith(fakePath),
        );
      });
    });
  });
});
